/**
 * Hunter Adams (vha3@cornell.edu)
 * 
 * HARDWARE CONNECTIONS
 *  - GPIO 16 ---> VGA Hsync
 *  - GPIO 17 ---> VGA Vsync
 *  - GPIO 18 ---> 330 ohm resistor ---> VGA Red
 *  - GPIO 19 ---> 330 ohm resistor ---> VGA Green
 *  - GPIO 20 ---> 330 ohm resistor ---> VGA Blue
 *  - RP2040 GND ---> VGA GND
 *
 * RESOURCES USED
 *  - PIO state machines 0, 1, and 2 on PIO instance 0
 *  - 4 DMA channels 
 *  - 2 x 153.6 kBytes of RAM (for pixel color data) double buffered
 *
 * Protothreads v1.4
 * Threads:
 * core 0:
 * -- Gravity and Graphics 
 * -- blink LED25 
 * core 1:
 * -- Serial i/o for restart
 */
// ==========================================
// === VGA graphics library
// ==========================================
#include "vga16_graphics_v3.h"
#include <stdio.h>
#include <stdlib.h> 
#include "pico/float.h"
#include <math.h>
#include "pico/stdlib.h"
#include "hardware/pio.h"
#include "hardware/dma.h"


// ==========================================
// === protothreads globals
// ==========================================
#include "hardware/sync.h"
#include "hardware/timer.h"
#include "pico/multicore.h"
#include "string.h"
// protothreads header
#include "pt_cornell_rp2040_v1_4.h"

#define n_mass 240
float x[n_mass], y[n_mass] ;
float x_1[n_mass], y_1[n_mass] ;
float ax[n_mass], ay[n_mass] ;
float m[n_mass] ;
short m_color[n_mass], m_size[n_mass] ;

float dt = 0.05f ;
float dt_sq ; // dt*dt
float tt = 0 ;
float eps = 1.0f ;

int restart_flag = false ;

// if this is defined then a big mass is injected from the left
#define disruptor
// for priunting
 char pstr[50] ;

// ==================================================
// === graphics demo -- RUNNING on core 0
// ==================================================
static PT_THREAD (protothread_graphics(struct pt *pt)) {
    PT_BEGIN(pt);
    static float tempx, tempy ;
    static int i, j;
    // the protothreads interval timer
    PT_INTERVAL_INIT() ;
    gpio_init(2);
    // ouptput
    gpio_set_dir(2, true);

    // Draw some filled rectangles
    fillRect(0, 0, 640, 480, BLACK); // erase
    fillRect(64, 0, 176, 50, BLUE); // blue box
    fillRect(250, 0, 176, 50, YELLOW); // red box
    fillRect(435, 0, 176, 50, GREEN); // green box

    // Write some text
    drawTextTiny8(65, 0, "Raspberry Pi Pico 2", WHITE, BLUE) ;
    drawTextGLCD(65, 10, "Floating Point Gravity demo", WHITE, BLUE) ;
    drawTextGLCD(65, 20, "Bruce Land brl4@cornell.edu", WHITE, BLUE) ;
    drawTextGLCD(65, 30, "Hunter Adams vha3@cornell.edu", WHITE, BLUE) ;
    //
    drawTextGLCD(438, 10, "Protothreads rp2040/2350 1.4", BLACK, GREEN) ;
    drawTextGLCD(438, 20, "CPU clock 150 Mhz", BLACK, GREEN) ;
    drawTextGLCD(438, 30, "vga16_v3 driver", BLACK, GREEN) ;
    drawTextGLCD(438, 40, "Double Buffer on rp2350 ONLY", BLACK, GREEN) ;
    //
    drawTextVGA437(260, 10, "Gravity Simulation", BLACK, YELLOW) ;
    //
    if (get_buffer_type() == 1)  drawTextGLCD(270, 30, "double buffer 60 fps", BLACK, YELLOW) ;
    if (get_buffer_type() == 2)  drawTextGLCD(270, 30, "double buffer 30 fps", BLACK, YELLOW) ;
    if (get_buffer_type() == 3)  drawTextGLCD(270, 30, "no double buffer    ", BLACK, YELLOW) ;
    sprintf(pstr, "number of particles %3d", n_mass) ;
    drawTextGLCD(270, 40, pstr, BLACK, YELLOW) ;  
     
    // dup the static content into both buffers
    copy_buffer_to_other();

    // init positions and velocities and masses
    // first, a heavy primary at rest
    m[0] = 8e2f ; x[0] = 320 ; y[0] = 260 ; x_1[0] =x[0] ; y_1[0] = y[0] ;
    m_color[0] = RED; m_size[0] = 3 ;

    // for circular orbit v = sqrt(GM/r)
    // around the primary
    for(int i=1; i<n_mass; i++){
      // init the masses, pos, vel
      m_color[i] = WHITE ;
      m_size[i] = 1 ;
      if (i < n_mass/2){
        m[i] = 0.001f ;
        x[i] = 320.0f + ((float)i + 50.0f) * ((rand() & 0x01)?1.0f :-1.0f) ;
        y[i] = 260.0f ; 
        x_1[i] = x[i] + 0.00f ;
        y_1[i] = y[i] + 1.4f/(sqrt(fabsf(x[i]-320))) * ((x[i] > 320)?1.0f :-1.0f) ;
      }
      else {
        m[i] = 0.001f ;
        x[i] = 320.0f ;
        y[i] = 260.0f + ((float)(i-n_mass/2) + 50.0f) * ((rand() & 0x01)?1.0f :-1.0f) ; 
        x_1[i] = x[i] + 1.4f/(sqrt(fabsf(y[i]-260.0f))) * ((y[i] > 260.0f)?-1.0f : 1.0f) ;
        y_1[i] = y[i] + 0.00f ;        
      }
       //printf("%f %f\n", x[i], y[i]);
    }
    
    dt_sq = dt * dt ;
    tt = 0 ;

    while(true) {
        // signal from serial thread to restart THIS thread
        if(restart_flag){
          restart_flag = false ;
          PT_RESTART(pt) ;
        }

        PT_YIELD_UNTIL(pt, draw_start_signal());
        //
        uint64_t start_time = PT_GET_TIME_usec();

        // clear takes 500 uSec
        clearLowFrame(50, BLACK) ;
        // update dynamic time
        tt += dt ;
        // zero the acceleration accumulators
        memset(ay, 0, sizeof(ay));
        memset(ax, 0, sizeof(ax));

        // inject a larger mass
        // eaarly inthe simulation
        #ifdef disruptor
         if ((int)tt == 1){
          // first erase the existing particle
         // fillCircle((short)x[75], (short)y[75], m_size[75], BLACK) ; 
          // interesting mass range is ~10 to 200
          m[75] =15.0f; x[75] = 150.0f;  y[75] = 320.0f ;
          x_1[75] =x[75] - 0.10f;  y_1[75] = 320.0f ;
          m_color[75] = GREEN ; m_size[75] = 2 ;
        }
        #endif
        
        // for each particle
        for(i=0; i<n_mass; i++){
          // compute accelerations
          // compute pairs so that can do this in half the time
          for(j=i+1; j<n_mass; j++){
            //
            //if(j != i){
              // compute distance between two particles
              float dx = x[j] - x[i] ;
              float dy = y[j] - y[i] ;
              // eps prevents divide by zero
              float dist_sq = dx*dx + dy*dy + eps ;
              // one over distance cubed
              // the 'cubed' includes the inverse sq law and
              // 1/dist -- which below will be combined with dx and dy
              // to form cos and sin components of the force
              float d  = 1/(sqrtf(dist_sq * dist_sq * dist_sq));
              // get the acceleration for each of the particles
              // (depends only on the mass of the OTHER particle)
              float si  =  m[j] * d ;
              float sj  =  m[i] * d ;
              // update accelerations for BOTh the particles with
              // opposite signs since the only difference is the
              // direction of the dx and dy vectors
              ax[i] += dx * si ;
              ax[j] -= dx * sj ;
              ay[i] += dy * si ;
              ay[j] -= dy * sj ;
            //}
          }
        
          // erase mass
          //fillCircle((short)x[i], (short)y[i], m_size[i], BLACK) ;

          // update pos using Verlet integration
          // https://en.wikipedia.org/wiki/Verlet_integration
          // for second order accuracy
          tempx = x[i] ;
          tempy = y[i] ;
          // x(t+dt) = 2*x(t) - x(t-dt) + a(t)*dt*dt
          // the form modified slightly to use +=
          // to save one floating add
          x[i] += x[i] - x_1[i] + ax[i] * dt_sq  ;
          y[i] += y[i] - y_1[i] + ay[i] * dt_sq ;
          x_1[i] = tempx ;
          y_1[i] = tempy ;
          
          // draw mass 
          fillCircle((short)x[i], (short)y[i], m_size[i], m_color[i]) ;     
        }
        // time documentation
       
        sprintf(pstr, "Dynamic Time = %5.1f", tt);
        drawTextTiny8(445, 450, pstr, GREEN, BLACK) ;

        setCursor(445, 30) ;
        sprintf(pstr, "Frame Time = %5.2f mSec", (float)(PT_GET_TIME_usec()-start_time)/1000.0f);
        drawTextTiny8(445, 460, pstr, GREEN, BLACK) ;

        //any other threads reaaady?
        PT_YIELD(pt);
       // 
    }
   PT_END(pt);
} // graphics thread

// ==================================================
// === toggle25 thread on core 0
// ==================================================
// the on-board LED blinks
static PT_THREAD (protothread_toggle25(struct pt *pt))
{
    PT_BEGIN(pt);
    static bool LED_state = false ;
    
     // set up LED p25 to blink
     gpio_init(25) ;	
     gpio_set_dir(25, GPIO_OUT) ;
     gpio_put(25, true);
     // data structure for interval timer
     PT_INTERVAL_INIT() ;

      while(1) {
        // yield time 0.1 second
        //PT_YIELD_usec(100000) ;
        PT_YIELD_INTERVAL(100000) ;

        // toggle the LED on PICO
        LED_state = LED_state? false : true ;
        gpio_put(25, LED_state);
        //
        // NEVER exit while
      } // END WHILE(1)
  PT_END(pt);
} // blink thread


// ==================================================
// === user's serial input thread on core 1
// ==================================================
// serial_read an serial_write do not block any thread
// except this one
static PT_THREAD (protothread_serial(struct pt *pt))
{
    PT_BEGIN(pt);
      static int test_in1, test_in2, sum ;
      //
      while(1) {
        // print prompt
        sprintf(pt_serial_out_buffer, "<enter> to restart: ");
        // spawn a thread to do the non-blocking write
        serial_write ;

        // spawn a thread to do the non-blocking serial read
         serial_read ;
         // tell the graphics thread to restart itself
         restart_flag = true ;
         
        // convert input string to number
        //sscanf(pt_serial_in_buffer,"%d %d", &test_in1, &test_in2) ;

        // add and convert back to sring
        //sprintf(pt_serial_out_buffer,"sum = %d\r\n", test_in1 + test_in2);
        // spawn a thread to do the non-blocking serial write
         //serial_write ;

        // NEVER exit while
      } // END WHILE(1)
  PT_END(pt);
} // serial thread

// ========================================
// === core 1 main -- started in main below
// ========================================
void core1_main(){ 
  //
  //  === add threads  ====================
  // for core 1
  pt_add_thread(protothread_serial) ;
  //
  // === initalize the scheduler ==========
  pt_schedule_start ;
  // NEVER exits
  // ======================================
}

// ========================================
// === core 0 main
// ========================================
int main(){
  // set the clock
  //set_sys_clock_khz(250000, true); // 171us
  // start the serial i/o
  stdio_init_all() ;
  // announce the threader version on system reset
  printf("\n\rProtothreads RP2040/2350 v1.4 two-core\n\r");

  // Initialize the VGA screen
  initVGA() ;
     
  // start core 1 threads
  multicore_reset_core1();
  multicore_launch_core1(&core1_main);

  // === config threads ========================
  // for core 0
  pt_add_thread(protothread_graphics);
  pt_add_thread(protothread_toggle25);
  //
  // === initalize the scheduler ===============
  pt_schedule_start ;
  // NEVER exits
  // ===========================================
} // end main